#include <ctime>
#include <vector>
#include <cstdlib>
#include <sstream>
#include <iostream>
#include <fstream>
#include <string>
#include <glpk.h>
#include <limits>
#include "src/JobTool.hpp"
#include "src/Distributor.hpp"
#include "src/MyMonitor.hpp"
#include "src/thrift/gen-cpp/Remote.h"
using namespace std;

FarmerInfo farmInfo;
Obj in;

int loadObj(string filename) {
	glp_prob *P = glp_create_prob();
	
	int err;
	// cplex
	err = glp_read_lp(P, NULL, filename.c_str());
	farmInfo.fileformat = "cplex";

	if (err != 0) {
		// modern mip
		err = glp_read_mps(P, GLP_MPS_FILE, NULL, filename.c_str());
		farmInfo.fileformat = "modern miplib";
	}
	if (err != 0) {
		// historic mip
		err = glp_read_mps(P, GLP_MPS_DECK, NULL, filename.c_str());
		farmInfo.fileformat = "historic miplib";
	}
	if (err != 0) return err;
	
	
	if (glp_get_obj_dir(P) == GLP_MAX) {
		in.isMax = true;
		JobTool::BAD_OptNBound = -1.0 * numeric_limits<double>::max();
		farmInfo.direction = "maximize";
	} else {
		in.isMax = false;
		JobTool::BAD_OptNBound = numeric_limits<double>::max();
		farmInfo.direction = "minimize";
	}
	
	int rows = glp_get_num_rows(P);
	int cols = glp_get_num_cols(P);
	farmInfo.cols    = cols;
	farmInfo.rows    = rows;
	farmInfo.notZero = glp_get_num_nz(P);

	for (int c = 1; c <= cols; ++c) {
		in.coefs.push_back(glp_get_obj_coef(P, c));

		Column col;
		col.at    = c;
		col.kind  = glp_get_col_kind(P, c);
		col.type  = glp_get_col_type(P, c);
		col.lower = glp_get_col_lb(P, c);
		col.upper = glp_get_col_ub(P, c);
		in.cols.push_back(col);
	}
	
	int rowcols;
	int rowIds[cols+1];
	double rowVals[cols+1];
	for (int r = 1; r <= rows; ++r) {
		Row row;
		row.at = r;
		row.type    = glp_get_row_type(P, r);
		row.lower   = glp_get_row_lb(P, r);
		row.upper   = glp_get_row_ub(P, r);
		rowcols     = glp_get_mat_row(P, r, rowIds, rowVals);
		for (int c = 0; c < rowcols; ++c) {
			row.ids.push_back(rowIds[c+1]);
			row.vals.push_back(rowVals[c+1]);
		}
		in.rows.push_back(row);
	}
	glp_delete_prob(P);
	return 0;
}

int main(int argc, char *argv[]) {
	if (argc < 3) {
		fprintf(
			stderr,
			"usage: %s <instance filename> <config file>\n\n"
			"Config-Format:\n"
			"<AnzahlWorker> \n"
			"<-1|job timeout(int ms)> \n"
			"<add +K (int) on every timeout/branch> \n"
			"<multiplicate this factor (float) on every timeout> \n"
			"<stop breadth-first search in depth bigger then (int %%)> \n"
			"<start presolver and stop usage of job timeout in a depth of (int %%) percent> \n"
			"<(int -1=no|10...) bestSync in ms> \n"
			"<logging every (int) ms> \n"
			"WorkerIpAddr1 Port1 WorkerIpAddr2 Port2 ...\n\n",
			argv[0]
		);
		return 1;
	}
	long long int dur;
	
	// conf ------------------------------------------------- go
	Configurator * conf = new Configurator();
	conf->filename = string(argv[1]); // instance-file !!
	ifstream file(argv[2]);
	string temp, rawdata;
	while (getline(file, temp)) rawdata += temp;
	file.close();
	stringstream f(rawdata);

	int workers;
	int intervallMs;
	f >> workers;
	f >> conf->jobTimeout;
	f >> conf->K;
	f >> conf->factor;
	f >> conf->breadthEndProcent;
	f >> conf->timeoutEndProcent;
	f >> conf->syncBest;
	f >> intervallMs;

	// conf ------------------------------------------------- end

	int err = loadObj(conf->filename);
	if (err != 0) {
		fprintf(stderr, "%s konnte nicht von glpk eingelesen werden.\n", argv[2]);
		return err;
	}
	
	MyMonitor * moni = new MyMonitor(intervallMs, farmInfo);
	Distributor distri(conf, moni);

	for (int i = 0; i < workers; ++i) {
		char ip[40]; //ipv6 + \0
		int port;
		f >> ip;
		f >> port;
		distri.push(ip, port);
		printf("%d/%d %s:%d\n", i+1, workers, ip, port);
	}

	dur = distri.run(&in);

	time_t now = time(0);
	struct tm tstruct;
	char stopTimeStr[80];
	tstruct = *localtime(&now);
	strftime(stopTimeStr, sizeof(stopTimeStr), "%Y%m%d-%H%M%S", &tstruct);

	ostringstream osConf;
	osConf << *(conf);

	printf(
		"Worker %d, Duration %lld s, Best %s, Now %s, %s, ",
		workers,
		dur,
		JobTool::toString(distri._result).c_str(),
		stopTimeStr,
		osConf.str().c_str()
	);

	vector<long> e = distri.getEnlargeCount();
	int s = e.size();
	printf("Timeouts ");

	for (int i=0; i < s; ++i)
		printf("%d:%ld ", i, e[i]);

	printf("\n");

	delete moni;
	delete conf;
	return 0;
}
